import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import Link from "next/link";
import { getSessionOptional } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { payPaymentIntent } from "@/lib/payments";
import { GuestPay } from "@/components/guest/GuestPay";

export default async function CheckoutPaymentIntentPage({ params }: { params: { token: string } }) {
  const session = await getSessionOptional();

  const pi = await prisma.paymentIntent.findUnique({
    where: { checkoutToken: params.token },
    include: { merchant: { include: { user: true } } },
  });

  async function pay() {
    "use server";
    if (!session) throw new Error("UNAUTHENTICATED");
    const res = await payPaymentIntent({ payerUserId: session.uid, checkoutToken: params.token });
    return res.txId;
  }

  if (!pi) {
    return (
      <>
        <TopNav />
        <Container>
          <div className="mx-auto max-w-xl">
            <Card>
              <CardHeader title="Checkout" subtitle="Not found" />
              <CardBody>
                <div className="text-sm text-zinc-700">This checkout link is invalid.</div>
              </CardBody>
            </Card>
          </div>
        </Container>
      </>
    );
  }

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="Checkout" subtitle={pi.merchant.tradingName} />
            <CardBody>
              <div className="rounded-xl border border-zinc-200 p-4">
                <div className="text-xs text-zinc-600">Amount</div>
                <div className="text-2xl font-semibold">P {fromMinor(pi.amount)}</div>
                <div className="mt-2 text-xs text-zinc-600">Reference: {pi.reference}</div>
                {pi.description ? <div className="mt-1 text-xs text-zinc-600">{pi.description}</div> : null}
                <div className="mt-3 text-xs text-zinc-500">Status: {pi.status}</div>
              </div>

              {pi.status === "SUCCEEDED" ? (
                <div className="mt-4 text-sm text-green-700">Payment completed.</div>
              ) : (
                <>
                  {session ? (
                    <form action={pay} className="mt-4">
                      <Button type="submit" className="w-full">Pay with LEFA</Button>
                    </form>
                  ) : (
                    <div className="mt-4 grid gap-2">
                      <Link href={`/signin?callbackUrl=/checkout/pi/${pi.checkoutToken}`}>
                        <Button className="w-full">Sign in to pay with LEFA</Button>
                      </Link>
                      <GuestPay mode="PAY_INTENT" checkoutToken={pi.checkoutToken} />
                    </div>
                  )}
                </>
              )}
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
